from pyparsing import Literal

from codeable_detectors.basic_detectors import AtLeastOneFileMatchesDetector
from codeable_detectors.evidences import Evidence, FailedEvidence, LinkEvidence
from codeable_detectors.pyparsing_patterns import round_braces_block, ID


class PHPConnectWithPDO(AtLeastOneFileMatchesDetector):
    def __init__(self, database_library):
        super().__init__()
        self.file_endings = ["php"]
        self.pdo_connect_pattern = Literal("=") + Literal("new") + Literal("PDO") + round_braces_block + Literal(";")
        self.db_lib_pattern = Literal(database_library) + Literal(":") + Literal("host") + Literal("=") + ID

    def detect_in_context(self, ctx, **kwargs):
        pdo_connect_matches = ctx.matches_pattern(self.pdo_connect_pattern)
        if pdo_connect_matches:
            db_lib_matches = ctx.matches_pattern(self.db_lib_pattern)
            if db_lib_matches:
                return Evidence(pdo_connect_matches + db_lib_matches)
            else:
                return FailedEvidence("no definition of database library host found")
        return FailedEvidence("no PDO connection found in php files")


class PHPMySQLPDOLink(PHPConnectWithPDO):
    def __init__(self):
        super().__init__("mysql")

    def detect_in_context(self, ctx, **kwargs):
        evidence = super().detect_in_context(ctx, **kwargs)
        if not evidence.has_succeeded():
            return evidence

        return LinkEvidence(evidence.matches).set_properties(detector_link_types=["mySQLProtocol"],
                                                             detector_technology_types=["php", "mysql"], kwargs=kwargs)
